param(
    [Parameter(Mandatory=$true)]
    [string]$HostName
)

# Script version for verification
$ScriptVersion = "1.0.7"
Write-Host "Running test-cloud-local-server.ps1 version $ScriptVersion" -ForegroundColor Cyan

# Trim whitespace and validate hostname
$HostName = $HostName.Trim()
if ([string]::IsNullOrWhiteSpace($HostName)) {
    Write-Host "Error: Hostname is missing or invalid." -ForegroundColor Red
    Write-Host "Usage: .\test-cloud-local-server.ps1 -HostName <IP_or_Hostname>" -ForegroundColor Yellow
    Write-Host "Example: .\test-cloud-local-server.ps1 -HostName 1.1.1.1" -ForegroundColor Yellow
    Write-Host "Description: This script checks your public IP and tests connections on ports 2222 (SSH), 27017 (MongoDB, HTTP 200/404 with body), and 8020 (web service, HTTP 200/404) of the specified host." -ForegroundColor Yellow
    exit 1
}

# Validate hostname as IP or domain
if (-not ($HostName -match "^(\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3})$|^([a-zA-Z0-9]([a-zA-Z0-9\-]{0,61}[a-zA-Z0-9])?\.)+[a-zA-Z]{2,}$")) {
    Write-Host "Error: '$HostName' is not a valid IP address or hostname." -ForegroundColor Red
    Write-Host "Usage: .\test-cloud-local-servert.ps1 -HostName <IP_or_Hostname>" -ForegroundColor Yellow
    Write-Host "Example: .\test-cloud-local-servert.ps1 -HostName 1.1.1.1" -ForegroundColor Yellow
    exit 1
}

# Debug: Output the hostname to confirm it’s being received
Write-Host "Checking ports on host: $HostName" -ForegroundColor Cyan

# Check public IP from multiple services
$services = @{
    "ipify" = @{Url = "https://api.ipify.org?format=json"; Field = "ip"}
    "ipapi" = @{Url = "https://ipapi.co/json/"; Field = "ip"}
    "jsonip" = @{Url = "https://jsonip.com"; Field = "ip"}
}

foreach ($service in $services.Keys) {
    try {
        $response = Invoke-WebRequest -Uri $services[$service].Url -Method Get -UseBasicParsing -TimeoutSec 5
        $json = $response.Content | ConvertFrom-Json
        $ip = $json.($services[$service].Field)
        Write-Host "Public IP from ${service}: $ip" -ForegroundColor Green
    } catch {
        Write-Host "Failed to get public IP from ${service}: Catch - $($_.Exception.Message)" -ForegroundColor Red
    }
}

# Port 2222: Check for SSH using raw TCP connection
try {
    Write-Host "Attempting to connect to ${HostName}:2222" -ForegroundColor Cyan
    $tcpClient = New-Object System.Net.Sockets.TcpClient
    $connectTask = $tcpClient.ConnectAsync($HostName, 2222)
    $wait = $connectTask.Wait(5000) # 5-second timeout
    if ($wait -and $tcpClient.Connected) {
        $stream = $tcpClient.GetStream()
        $reader = New-Object System.IO.StreamReader($stream)
        $banner = $reader.ReadLine()
        $tcpClient.Close()
        if ($banner -like "SSH-2.0-*") {
            Write-Host "Port 2222: Success - SSH service detected ($banner)" -ForegroundColor Green
            Write-Host "OK" -ForegroundColor Green
        } else {
            # Try HTTP as a fallback
            try {
                $response = Invoke-WebRequest -Uri "http://$($HostName):2222/" -Method Get -UseBasicParsing -TimeoutSec 5 -ErrorAction SilentlyContinue
                if ($response -and ($response.StatusCode -eq 200 -or $response.StatusCode -eq 404)) {
                    Write-Host "Port 2222: HTTP Success - Status $($response.StatusCode)" -ForegroundColor Green
                    Write-Host "OK" -ForegroundColor Green
                } else {
                    Write-Host "Port 2222: Failure - Unexpected HTTP status $(if ($response) { $response.StatusCode } else { 'No response' })" -ForegroundColor Red
                    Write-Host "FAILED" -ForegroundColor Red
                }
            } catch {
                Write-Host "Port 2222: Failure - Non-SSH, non-HTTP service detected: Catch - $($_.Exception.Message)" -ForegroundColor Red
                Write-Host "FAILED" -ForegroundColor Red
            }
        }
    } else {
        Write-Host "Port 2222: Failure - Port is closed or unreachable: Catch - Connection failed" -ForegroundColor Red
        Write-Host "FAILED" -ForegroundColor Red
        if ($tcpClient) { $tcpClient.Close() }
    }
} catch {
    Write-Host "Port 2222: Failure - Catch - $($_.Exception.Message)" -ForegroundColor Red
    Write-Host "FAILED" -ForegroundColor Red
    if ($tcpClient) { $tcpClient.Close() }
}

# Port 27017: Check for MongoDB (HTTP 200/404 with body)
try {
    $uri = "http://$($HostName):27017/"
    Write-Host "Attempting to connect to $uri" -ForegroundColor Cyan
    $response = Invoke-WebRequest -Uri $uri -Method Get -UseBasicParsing -TimeoutSec 5 -ErrorAction SilentlyContinue
    if ($response -and ($response.StatusCode -eq 200 -or $response.StatusCode -eq 404)) {
        if ($response.Content.Length -gt 0) {
            # Check for MongoDB-specific content
            if ($response.Content -match "MongoDB" -or $response.Content -match "db" -or $response.Content -match "version") {
                Write-Host "Port 27017: Success - MongoDB detected (Status $($response.StatusCode) with body)" -ForegroundColor Green
                Write-Host "OK" -ForegroundColor Green
            } else {
                Write-Host "Port 27017: Success - Status $($response.StatusCode) with body, but not MongoDB" -ForegroundColor Yellow
                Write-Host "OK" -ForegroundColor Green
            }
        } else {
            Write-Host "Port 27017: Failure - Status $($response.StatusCode) but empty body" -ForegroundColor Red
            Write-Host "FAILED" -ForegroundColor Red
        }
    } else {
        Write-Host "Port 27017: Failure - Catch - $(if ($response) { "Unexpected status $($response.StatusCode)" } else { "No response received: $($_.Exception.Message)" })" -ForegroundColor Red
        Write-Host "FAILED" -ForegroundColor Red
    }
} catch {
    Write-Host "Port 27017: Failure - Catch - $($_.Exception.Message)" -ForegroundColor Red
    Write-Host "FAILED" -ForegroundColor Red
}

# Port 8020: Web service (HTTP 200/404, body optional)
try {
    $uri = "http://$($HostName):8020/"
    Write-Host "Attempting to connect to $uri" -ForegroundColor Cyan
    $request = [System.Net.HttpWebRequest]::Create($uri)
    $request.Method = "GET"
    $request.Timeout = 5000 # 5-second timeout
    try {
        $response = $request.GetResponse()
        $statusCode = [int]$response.StatusCode
        $response.Close()
        Write-Host "Port 8020: Debug - Response received, StatusCode: $statusCode" -ForegroundColor Yellow
        if ($statusCode -eq 200 -or $statusCode -eq 404) {
            Write-Host "Port 8020: Success - Web service detected (Status $statusCode)" -ForegroundColor Green
            Write-Host "OK" -ForegroundColor Green
        } else {
            Write-Host "Port 8020: Failure - Unexpected status $statusCode" -ForegroundColor Red
            Write-Host "FAILED" -ForegroundColor Red
        }
    } catch {
        $statusCode = $null
        if ($_.Exception.InnerException -is [System.Net.WebException]) {
            $webResponse = $_.Exception.InnerException.Response
            if ($webResponse) {
                $statusCode = [int]$webResponse.StatusCode
                $webResponse.Close()
                Write-Host "Port 8020: Debug - Response received, StatusCode: $statusCode" -ForegroundColor Yellow
                if ($statusCode -eq 200 -or $statusCode -eq 404) {
                    Write-Host "Port 8020: Success - Web service detected (Status $statusCode)" -ForegroundColor Green
                    Write-Host "OK" -ForegroundColor Green
                } else {
                    Write-Host "Port 8020: Failure - Unexpected status $statusCode" -ForegroundColor Red
                    Write-Host "FAILED" -ForegroundColor Red
                }
            } else {
                Write-Host "Port 8020: Failure - Inner Catch - No response received: $($_.Exception.Message)" -ForegroundColor Red
                Write-Host "FAILED" -ForegroundColor Red
            }
        } else {
            Write-Host "Port 8020: Failure - Inner Catch - $($_.Exception.Message)" -ForegroundColor Red
            Write-Host "FAILED" -ForegroundColor Red
        }
    }
} catch {
    Write-Host "Port 8020: Failure - Outer Catch - $($_.Exception.Message)" -ForegroundColor Red
    Write-Host "FAILED" -ForegroundColor Red
}